<?php declare(strict_types=1);

namespace SupportPal\Marketplace\Http\Api\Addons;

use SupportPal\Marketplace\Exceptions\AccountNotFound;
use SupportPal\Marketplace\Exceptions\AddonNotFound;
use SupportPal\Marketplace\Exceptions\AddonUnavailable;
use SupportPal\Marketplace\Exceptions\ApiException;
use SupportPal\Marketplace\Exceptions\CancelFailed;
use SupportPal\Marketplace\Http\Request;

use function sprintf;

class Cancel
{
    /** @var Request */
    protected Request $api;

    /**
     * @param Request $api
     */
    public function __construct(Request $api)
    {
        $this->api = $api;
    }

    /**
     * @throws AccountNotFound
     * @throws AddonNotFound
     * @throws AddonUnavailable
     * @throws CancelFailed
     * @throws ApiException
     */
    public function cancel(string $uuid, int $id, int $addon_id): bool
    {
        try {
            $response = $this->api->post(sprintf('addons/cancel/%d', $id), [
                'uuid'     => $uuid,
                'addon_id' => $addon_id,
            ]);
        } catch (ApiException $e) {
            if ($e->getCode() === 404) {
                if ($e->getMessage() === 'Unable to find account for given UUID.') {
                    throw new AccountNotFound($e->getMessage());
                }

                if ($e->getMessage() === 'Add-on is unavailable.') {
                    throw new AddonUnavailable($e->getMessage());
                }

                throw new AddonNotFound($e->getMessage());
            }

            if ($e->getCode() === 422) {
                if ($e->getMessage() === 'Could not terminate add-on.') {
                    throw new CancelFailed($e->getMessage());
                }
            }

            throw $e;
        }

        if (! isset($response->status)) {
            throw new CancelFailed('Invalid response from marketplace.');
        }

        return $response->status === 'success';
    }
}
