<?php declare(strict_types=1);

namespace SupportPal\Marketplace\Http\Api\Addons;

use SupportPal\Marketplace\Exceptions\AccountNotFound;
use SupportPal\Marketplace\Exceptions\AddonNotFound;
use SupportPal\Marketplace\Exceptions\AddonUnavailable;
use SupportPal\Marketplace\Exceptions\ApiException;
use SupportPal\Marketplace\Exceptions\LicenseAlreadyHasAddon;
use SupportPal\Marketplace\Exceptions\LicenseNotFound;
use SupportPal\Marketplace\Exceptions\OrderFailed;
use SupportPal\Marketplace\Exceptions\OrderPaymentRequired;
use SupportPal\Marketplace\Http\Request;

use function sprintf;

class Order
{
    /** @var Request */
    protected Request $api;

    /**
     * @param Request $api
     */
    public function __construct(Request $api)
    {
        $this->api = $api;
    }

    /**
     * @throws AccountNotFound
     * @throws AddonNotFound
     * @throws AddonUnavailable
     * @throws ApiException
     * @throws LicenseAlreadyHasAddon
     * @throws LicenseNotFound
     * @throws OrderFailed
     * @throws OrderPaymentRequired
     */
    public function order(string $uuid, string $license, int $id, ?string $billingCycle = null): bool
    {
        try {
            $response = $this->api->post(sprintf('addons/order/%d', $id), [
                'uuid'          => $uuid,
                'license'       => $license,
                'billing_cycle' => $billingCycle,
            ]);
        } catch (ApiException $e) {
            if ($e->getCode() === 404) {
                if ($e->getMessage() === 'Unable to find account for given UUID.') {
                    throw new AccountNotFound($e->getMessage());
                }

                if ($e->getMessage() === 'Unable to find license under account.'
                    || $e->getMessage() === 'License not found.'
                ) {
                    throw new LicenseNotFound($e->getMessage());
                }

                if ($e->getMessage() === 'Add-on is unavailable.') {
                    throw new AddonUnavailable($e->getMessage());
                }

                throw new AddonNotFound($e->getMessage());
            }

            if ($e->getCode() === 422) {
                if ($e->getMessage() === 'License already has add-on.') {
                    throw new LicenseAlreadyHasAddon($e->getMessage());
                }

                if ($e->getMessage() === 'Order failed.') {
                    throw new OrderFailed($e->getMessage());
                }
            }

            throw $e;
        }

        if (! isset($response->status)) {
            throw new OrderFailed('Invalid response from marketplace.');
        }

        if ($response->status === 'success') {
            if (isset($response->message)) {
                throw new OrderPaymentRequired($response->message);
            }

            return true;
        }

        return false;
    }
}
