<?php declare(strict_types=1);

namespace SupportPal\Marketplace\Http;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\BadResponseException;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Exception\TransferException;
use GuzzleHttp\HandlerStack;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Kevinrob\GuzzleCache\CacheMiddleware;
use Kevinrob\GuzzleCache\Storage\LaravelCacheStorage;
use Kevinrob\GuzzleCache\Strategy\PublicCacheStrategy;
use SupportPal\Marketplace\Exceptions\ApiException;
use SupportPal\ProxyService\ProxyService;

use function env;
use function fclose;
use function fopen;
use function json_decode;
use function rewind;
use function sprintf;
use function stream_get_contents;

class Request
{
    private ProxyService $proxyService;

    public function __construct(ProxyService $proxyService)
    {
        $this->proxyService = $proxyService;
    }

    /**
     * Domains where we can access the marketplace Api.
     * Trailing slash is required for Guzzle 6 base_uri setting to work with directory paths.
     */
    public function getBaseUrl(): string
    {
        return env('MARKETPLACE_URL', 'https://marketplace.supportpal.com') . '/api/';
    }

    /**
     * Perform a GET http request.
     *
     * @param  mixed[]  $parameters
     * @return mixed
     * @throws ApiException
     */
    public function get(string $url, array $parameters = [])
    {
        return $this->call('GET', $url, $parameters);
    }

    /**
     * Perform a GET http request.
     *
     * @param  mixed[]  $parameters
     * @return mixed
     * @throws ApiException
     */
    public function post(string $url, array $parameters = [])
    {
        return $this->call('POST', $url, $parameters);
    }

    /**
     * @param string $url
     * @param mixed[] $parameters
     * @return mixed
     * @throws ApiException
     */
    private function call(string $method, string $url, array $parameters)
    {
        $fp = @ fopen('php://temp', 'w+');

        $paramOption = 'form_params';
        if ($method === 'GET') {
            $paramOption = 'query';
        }

        try {
            $response = $this->client()->request($method, $url, [$paramOption => $parameters, 'debug' => $fp]);

            return json_decode((string) $response->getBody());
        } catch (TransferException $e) {
            Log::error(sprintf('Marketplace Api error: \'%s\'.', $e->getMessage()), [
                'response' => $e instanceof RequestException ? $e->getResponse() : null,
                'request'  => $fp ? $this->getStreamContent($fp) : null
            ]);

            if ($e instanceof BadResponseException) {
                $response = json_decode((string) $e->getResponse()->getBody());

                if (isset($response->status) && $response->status === 'error' && ! empty($response->message)) {
                    throw new ApiException($response->message, $e->getCode(), $e);
                }
            }

            throw new ApiException($e->getMessage(), $e->getCode(), $e);
        } finally {
            if ($fp) {
                @ fclose($fp);
            }
        }
    }

    /**
     * Get the contents from a stream.
     *
     * @param  resource $resource
     * @return false|string
     */
    protected function getStreamContent($resource)
    {
        rewind($resource);

        return stream_get_contents($resource);
    }

    protected function connectionTimeout(): int
    {
        return (int) Config::get('app.marketplace.connect_timeout', 10);
    }

    protected function timeout(): int
    {
        return (int) Config::get('app.marketplace.timeout', 10);
    }

    private function client(): Client
    {
        $stack = HandlerStack::create();
        $stack->push(new CacheMiddleware(
            new PublicCacheStrategy(
                new LaravelCacheStorage(Cache::store())
            )
        ), 'cache');

        return $this->proxyService->createGuzzle([
            'handler'         => $stack,
            'base_uri'        => $this->getBaseUrl(),
            'connect_timeout' => $this->connectionTimeout(),
            'timeout'         => $this->timeout(),
            'headers' => [
                'User-Agent' => 'SupportPal Marketplace Client (https://www.supportpal.com)',
                'Accept'     => 'application/json'
            ]
        ]);
    }
}
