<?php declare(strict_types=1);

namespace SupportPal\OAuth\Manager;

use Illuminate\Support\Arr;
use Illuminate\Support\Manager;
use InvalidArgumentException;
use SupportPal\OAuth\Events\CreatedProvider;
use SupportPal\OAuth\Exceptions;
use SupportPal\OAuth\Provider\Facebook\Facebook;
use SupportPal\OAuth\Provider\Facebook\Server as FacebookProvider;
use SupportPal\OAuth\Provider\Google\Google;
use SupportPal\OAuth\Provider\Google\Server as GoogleProvider;
use SupportPal\OAuth\Provider\Microsoft\Microsoft;
use SupportPal\OAuth\Provider\Microsoft\Server as MicrosoftProvider;
use SupportPal\OAuth\Provider\Provider;
use SupportPal\OAuth\Provider\Steam\Server as SteamServer;
use SupportPal\OAuth\Provider\Steam\Steam;
use SupportPal\OAuth\Provider\Twitter\Server;
use SupportPal\OAuth\Provider\Twitter\Twitter;
use SupportPal\OAuth\Provider\Whmcs\Server as WhmcsProvider;
use SupportPal\OAuth\Provider\Whmcs\Whmcs;

use function route;
use function sprintf;

class OAuth extends Manager implements Factory
{
    /**
     * List of callback actions.
     *
     * @var Action[]
     */
    protected $actions = [];

    /**
     * Register an action.
     *
     * @param string $name
     * @param Action $callback
     * @return $this
     */
    public function setAction(string $name, Action $callback)
    {
        $this->actions[$name] = $callback;

        return $this;
    }

    /**
     * Get an action by name.
     *
     * @param string $name
     * @return Action
     */
    public function getAction(string $name)
    {
        if (! isset($this->actions[$name])) {
            throw new Exceptions\InvalidArgumentException(sprintf('Action with \'%s\' has not been registered.', $name));
        }

        return $this->actions[$name];
    }

    /**
     * Get redirect URI for provider.
     *
     * @param string $provider
     * @return string
     */
    public function redirectUri(string $provider)
    {
        return route('vendor.supportpal.oauth.callback', ['provider' => $provider]);
    }

    /**
     * Get a driver instance.
     *
     * @param  string  $driver
     * @return Provider
     *
     * @throws InvalidArgumentException
     */
    public function driver($driver = null)
    {
        return parent::driver($driver);
    }

    /**
     * Create a new driver instance.
     *
     * @param  string  $driver
     * @return Provider
     *
     * @throws InvalidArgumentException
     */
    protected function createDriver($driver)
    {
        $provider = parent::createDriver($driver);

        CreatedProvider::dispatch($provider);

        return $provider;
    }

    /**
     * Get the default driver name.
     *
     * @return string
     */
    public function getDefaultDriver()
    {
        throw new InvalidArgumentException('Default driver is not supported. Specify a valid driver.');
    }

    /**
     * Create an instance of the specified driver.
     *
     * @return Facebook
     */
    protected function createFacebookDriver()
    {
        $name = 'facebook';
        $config = $this->container['config']->get(sprintf('services.%s', $name));

        $provider = $this->container->make(FacebookProvider:: class, [
            'options' => [
                'clientId'          => Arr::get($config, 'client_id'),
                'clientSecret'      => Arr::get($config, 'client_secret'),
                'redirectUri'       => $this->redirectUri($name),
                'graphApiVersion'   => 'v6.0',
                'fields'            => [
                    'id', 'name', 'email', 'gender', 'verified', 'link',
                    'picture.type(large){url,is_silhouette}',
                ],
            ],
        ]);

        return $this->container->make(Facebook::class, [
            'server'  => $provider
        ]);
    }

    /**
     * Create an instance of the specified driver.
     *
     * @return Google
     */
    protected function createGoogleDriver()
    {
        $name = 'google';
        $config = $this->container['config']->get(sprintf('services.%s', $name));

        $provider = $this->container->make(GoogleProvider::class, [
            'options' => [
                'clientId'     => Arr::get($config, 'client_id'),
                'clientSecret' => Arr::get($config, 'client_secret'),
                'redirectUri'  => $this->redirectUri($name),
            ],
        ]);

        return $this->container->make(Google::class, [
            'server'  => $provider
        ]);
    }

    /**
     * Create an instance of the specified driver.
     *
     * @return Microsoft
     */
    protected function createMicrosoftDriver()
    {
        $name = 'microsoft';
        $config = $this->container['config']->get(sprintf('services.%s', $name));

        $provider = $this->container->make(MicrosoftProvider::class, [
            'options' => [
                'clientId'     => Arr::get($config, 'client_id'),
                'clientSecret' => Arr::get($config, 'client_secret'),
                'redirectUri'  => $this->redirectUri($name),
            ],
        ]);

        return $this->container->make(Microsoft::class, [
            'server'  => $provider
        ]);
    }

    /**
     * Create an instance of the specified driver.
     *
     * @return Steam
     */
    protected function createSteamDriver()
    {
        $name = 'steam';
        $config = $this->container['config']->get(sprintf('services.%s', $name));

        $server = $this->container->make(SteamServer::class, [
            'options' => [
                'apiKey'       => Arr::get($config, 'client_secret'),
                'redirectUri'  => $this->redirectUri($name),
            ],
        ]);

        return $this->container->make(Steam::class, [
            'server'  => $server,
        ]);
    }

    /**
     * Create an instance of the specified driver.
     *
     * @return Twitter
     */
    protected function createTwitterDriver()
    {
        $name = 'twitter';
        $config = $this->container['config']->get(sprintf('services.%s', $name));

        $provider = $this->container->make(Server::class, [
            'clientCredentials' => [
                'identifier'   => Arr::get($config, 'client_id'),
                'secret'       => Arr::get($config, 'client_secret'),
                'callback_uri' => $this->redirectUri($name),
            ],
        ]);

        return $this->container->make(Twitter::class, [
            'server'  => $provider
        ]);
    }

    /**
     * Create an instance of the specified driver.
     *
     * @return Whmcs
     */
    protected function createWhmcsDriver()
    {
        $name = 'whmcs';
        $config = $this->container['config']->get(sprintf('services.%s', $name));

        $provider = $this->container->make(WhmcsProvider::class, [
            'options' => [
                'url'          => Arr::get($config, 'url'),
                'clientId'     => Arr::get($config, 'client_id'),
                'clientSecret' => Arr::get($config, 'client_secret'),
                'redirectUri'  => $this->redirectUri($name),
            ],
        ]);

        return $this->container->make(Whmcs::class, [
            'server'  => $provider
        ]);
    }
}
