<?php declare(strict_types=1);

/**
 * File Server.php
 *
 * Adapted from https://github.com/stevenmaguire/oauth2-microsoft
 *
 * @copyright  Copyright (c) 2015-2020 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\OAuth\Provider\Microsoft;

use League\OAuth2\Client\Provider\AbstractProvider;
use League\OAuth2\Client\Provider\Exception\IdentityProviderException;
use League\OAuth2\Client\Token\AccessToken;
use Psr\Http\Message\ResponseInterface;
use SupportPal\OAuth\Provider\Server as Contract;

/**
 * Class Server
 */
class Server extends AbstractProvider implements Contract
{
    /**
     * Get authorization url to begin OAuth flow
     *
     * @return string
     */
    public function getBaseAuthorizationUrl()
    {
        return 'https://login.microsoftonline.com/common/oauth2/v2.0/authorize';
    }

    /**
     * Get access token url to retrieve token
     *
     * @param mixed[] $params
     * @return string
     */
    public function getBaseAccessTokenUrl(array $params)
    {
        return 'https://login.microsoftonline.com/common/oauth2/v2.0/token';
    }

    /**
     * Get default scopes
     *
     * @return array
     */
    protected function getDefaultScopes()
    {
        return ['User.Read'];
    }

    /**
     * Check a provider response for errors.
     *
     * @param ResponseInterface $response
     * @param mixed[] $data
     * @return void
     * @throws IdentityProviderException
     */
    protected function checkResponse(ResponseInterface $response, $data)
    {
        if (isset($data['error'])) {
            throw new IdentityProviderException(
                $data['error_description'] ?? $response->getReasonPhrase(),
                $response->getStatusCode(),
                $response
            );
        }
    }

    /**
     * Generate a user object from a successful user details request.
     *
     * @param mixed[] $response
     * @param AccessToken $token
     * @return MicrosoftResourceOwner
     */
    protected function createResourceOwner(array $response, AccessToken $token)
    {
        return new MicrosoftResourceOwner($response);
    }

    /**
     * Get provider url to fetch user details
     *
     * @param  AccessToken $token
     *
     * @return string
     */
    public function getResourceOwnerDetailsUrl(AccessToken $token)
    {
        return 'https://graph.microsoft.com/v1.0/me';
    }

    /**
     * Returns the string that should be used to separate scopes when building
     * the URL for requesting an access token.
     *
     * @return string Scope separator, defaults to ','
     */
    protected function getScopeSeparator()
    {
        return ' ';
    }

    /**
     * Returns the authorization headers used by this provider.
     *
     * Typically this is "Bearer" or "MAC". For more information see:
     * http://tools.ietf.org/html/rfc6749#section-7.1
     *
     * No default is provided, providers must overload this method to activate
     * authorization headers.
     *
     * @param  AccessToken|null $token Either a string or an access token instance
     * @return array
     */
    protected function getAuthorizationHeaders($token = null)
    {
        return ['Authorization' => 'Bearer ' . $token];
    }

    /**
     * Set client ID.
     *
     * @param string $id
     * @return $this
     */
    public function setClientId(string $id)
    {
        $this->clientId = $id;

        return $this;
    }

    /**
     * Set client secret.
     *
     * @param string $secret
     * @return $this
     */
    public function setClientSecret(string $secret)
    {
        $this->clientSecret = $secret;

        return $this;
    }

    /**
     * Set redirect URI.
     *
     * @param string $redirectUri
     * @return $this
     */
    public function setRedirectUri(string $redirectUri)
    {
        $this->redirectUri = $redirectUri;

        return $this;
    }

    /**
     * Builds the authorization URL.
     *
     * @param string[] $options
     * @return string
     */
    public function getAuthorisationUrl(array $options = [])
    {
        return $this->getAuthorizationUrl($options);
    }
}
