<?php declare(strict_types=1);

/**
 * File Provider.php
 *
 * @copyright  Copyright (c) 2015-2020 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\OAuth\Provider;

use GuzzleHttp\Client;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use SupportPal\OAuth\User\User;

use function array_merge;
use function bin2hex;
use function random_bytes;
use function redirect;
use function strlen;

/**
 * Class Provider
 */
abstract class Provider
{
    /**
     * Session keys.
     */
    const ACTION_SESSION_KEY = 'oauth.action';
    const STATE_SESSION_KEY = 'oauth.state';

    /**
     * Http request instance.
     *
     * @var Request
     */
    protected $request;

    /**
     * League OAuth instance.
     *
     * @var mixed
     */
    protected $server;

    /**
     * Provider constructor.
     *
     * @param Request $request
     * @param Server $server
     */
    public function __construct(Request $request, Server $server)
    {
        $this->request = $request;
        $this->server = $server;
    }

    /**
     * Redirect to the OAuth provider.
     *
     * @param string $action Name of an action.
     * @param array<string, mixed[]> $options
     * @return RedirectResponse
     */
    public function redirect(string $action, array $options = [])
    {
        // Store the name of the action we want to call after the OAuth flow.
        $this->storeAction($action);

        // Store state in the session to prevent CSRF attacks.
        $state = $this->storeState();

        // Second part of OAuth 2.0 authentication is to redirect the
        // resource owner to the login screen on the server.
        $url = $this->getServer()->getAuthorisationUrl(
            array_merge($options, ['state' => $state])
        );

        return redirect()->to($url);
    }

    /**
     * Get the User instance for the authenticated user.
     *
     * @return User
     */
    abstract public function user();

    /**
     * Set the Guzzle HTTP client.
     *
     * @param Client $client
     * @return $this
     */
    public function setHttpClient(Client $client)
    {
        $this->getServer()->setHttpClient($client);

        return $this;
    }

    /**
     * League OAuth server instance.
     *
     * @return Server
     */
    public function getServer()
    {
        return $this->server;
    }

    /**
     * Store the name of the action we want to call after the OAuth flow.
     *
     * @param string $action
     * @return $this
     */
    protected function storeAction(string $action)
    {
        $this->request->session()->put(static::ACTION_SESSION_KEY, $action);

        return $this;
    }

    /**
     * Store the OAuth state (CSRF token) in the session.
     *
     * @return string
     */
    protected function storeState()
    {
        // Store state in the session to prevent CSRF attacks.
        $this->request->session()->put(static::STATE_SESSION_KEY, $state = $this->getRandomState());

        return $state;
    }

    /**
     * Determine if the current request / session has a mismatching "state".
     *
     * @return bool
     */
    protected function hasInvalidState()
    {
        $state = $this->request->session()->pull(static::STATE_SESSION_KEY);

        return ! (strlen($state) > 0 && $this->request->input('state') === $state);
    }

    /**
     * Returns a new random string to use as the state parameter in an
     * authorization flow.
     *
     * @param  int $length Length of the random string to be generated.
     * @return string
     */
    protected function getRandomState($length = 32)
    {
        // Converting bytes to hex will always double length. Hence, we can reduce
        // the amount of bytes by half to produce the correct length.
        return bin2hex(random_bytes($length / 2));
    }
}
