<?php declare(strict_types=1);

/**
 * File Nonce.php
 *
 * @copyright  Copyright (c) 2015-2020 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\OAuth\Provider\Steam;

use DateTime;
use Illuminate\Contracts\Cache\Repository;
use SupportPal\OAuth\Exceptions\RuntimeException;
use Throwable;

use function count;
use function now;
use function preg_match;
use function sprintf;

/**
 * Class Nonce
 */
class Nonce
{
    /**
     * Cache instance.
     *
     * @var Repository
     */
    protected $cache;

    /**
     * Nonce.
     *
     * @var string
     */
    protected $nonce;

    /**
     * Range (in seconds) that the nonce timestamp can be considered valid.
     *
     * @var int
     */
    protected $skew = 60;

    /**
     * Nonce constructor.
     *
     * @param Repository $cache
     * @param string $nonce
     */
    public function __construct(Repository $cache, string $nonce)
    {
        $this->cache = $cache;
        $this->nonce = $nonce;
    }

    /**
     * Check whether the nonce is valid.
     *
     * @return bool
     */
    public function isValid(): bool
    {
        $this->hasExpired();
        $this->cache();

        return true;
    }

    /**
     * The time-stamp MAY be used to reject responses that are too far away from the current time.
     *
     * @return bool
     * @throws RuntimeException
     */
    public function hasExpired(): bool
    {
        $time = $this->timestamp();

        if ($time <= now()->subSeconds($this->skew)->getTimestamp()) {
            throw new RuntimeException(sprintf('Timestamp from nonce is earlier then time() - %ds', $this->skew));
        }

        if ($time >= now()->addSeconds($this->skew)->getTimestamp()) {
            throw new RuntimeException(sprintf('Timestamp from nonce is older then time() + %ds', $this->skew));
        }

        return false;
    }

    /**
     * Extract the timestamp from the nonce.
     *
     * @return int
     * @throws RuntimeException
     */
    public function timestamp(): int
    {
        $result = preg_match('/(\d{4})-(\d\d)-(\d\d)T(\d\d):(\d\d):(\d\d)Z(.*)/', $this->nonce, $matches);
        if ($result !== 1 || count($matches) !== 8) {
            throw new RuntimeException(sprintf('Unexpected nonce format \'%s\'', $this->nonce));
        }

        [,$year, $month, $day, $hour, $min, $sec] = $matches;

        try {
            $timestamp = new DateTime;
            $timestamp->setDate((int) $year, (int) $month, (int) $day);
            $timestamp->setTime((int) $hour, (int) $min, (int) $sec);
        } catch (Throwable $e) {
            throw new RuntimeException('Timestamp from nonce is not valid', $e->getCode(), $e);
        }

        return $timestamp->getTimestamp();
    }

    /**
     * Cache the nonce to prevent replay attacks.
     *
     * @return void
     * @throws RuntimeException
     */
    protected function cache(): void
    {
        if ($this->cache->has($this->nonce)) {
            throw new RuntimeException(sprintf("Nonce '%s' cannot be reused.", $this->nonce));
        }

        $this->cache->remember($this->nonce, $this->skew, function () {
            return $this->nonce;
        });
    }
}
