<?php declare(strict_types=1);

/**
 * File Server.php
 *
 * @copyright  Copyright (c) 2015-2020 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\OAuth\Provider\Steam;

use GuzzleHttp\Client as HttpClient;
use GuzzleHttp\ClientInterface as Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\RequestOptions;
use Illuminate\Contracts\Cache\Repository;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use SimpleXMLElement;
use SupportPal\OAuth\Exceptions\InvalidArgumentException;
use SupportPal\OAuth\Exceptions\RuntimeException;
use SupportPal\OAuth\Provider\Server as Contract;

use function http_build_query;
use function is_numeric;
use function json_decode;
use function preg_match;
use function sprintf;
use function str_replace;
use function strpos;
use function vsprintf;

/**
 * Class Server
 */
class Server implements Contract
{
    /**
     * Cache repository.
     *
     * @var Repository
     */
    protected $cache;

    /**
     * Http request.
     *
     * @var Request
     */
    protected $request;

    /**
     * API key.
     *
     * @var string
     */
    protected $apiKey;

    /**
     * Redirect URI.
     *
     * @var string
     */
    protected $redirectUri;

    /**
     * Guzzle Http client.
     *
     * @var Client
     */
    protected $client;

    /**
     * Server constructor.
     *
     * @param Request $request
     * @param Repository $cache
     * @param string[] $options
     */
    public function __construct(Request $request, Repository $cache, array $options)
    {
        $this->cache = $cache;
        $this->request = $request;

        if (empty($options['apiKey'])) {
            throw new InvalidArgumentException('apiKey is required.');
        }

        if (empty($options['redirectUri'])) {
            throw new InvalidArgumentException('redirectUri is required.');
        }

        $this->apiKey = $options['apiKey'];
        $this->redirectUri = $options['redirectUri'];
    }

    /**
     * Build the Steam login URL.
     *
     * @param string[] $options
     * @return string
     * @throws GuzzleException
     */
    public function getAuthorisationUrl(array $options = [])
    {
        $entryPoint = $this->discover('https://steamcommunity.com/openid');

        $params = [
            'openid.ns'         => 'http://specs.openid.net/auth/2.0',
            'openid.mode'       => 'checkid_setup',
            'openid.return_to'  => $this->redirectUri,
            'openid.realm'      => $this->redirectUri,
            // For the purposes of making OpenID Authentication requests, the value
            // "http://specs.openid.net/auth/2.0/identifier_select" MUST be used as both the Claimed Identifier
            // and the OP-Local Identifier when an OP Identifier is entered.
            'openid.identity'   => 'http://specs.openid.net/auth/2.0/identifier_select',
            'openid.claimed_id' => 'http://specs.openid.net/auth/2.0/identifier_select',
        ];

        return $entryPoint . '?' . http_build_query($params, '', '&');
    }

    /**
     * Returns the URL for requesting the resource owner's details.
     *
     * @return array
     * @throws GuzzleException
     * @throws RuntimeException
     */
    public function getResourceOwner()
    {
        $this->verifyAssertion();

        $claimedId = $this->request->get('openid_claimed_id');
        $url       = 'https://api.steampowered.com/ISteamUser/GetPlayerSummaries/v0002/';

        $response = $this->getHttpClient()->request('GET', $url, [
            RequestOptions::QUERY => [
                'key'      => $this->apiKey,
                'steamids' => $this->getUserId($claimedId)
            ]
        ]);

        $contents = json_decode($response->getBody()->getContents(), true);

        return Arr::get($contents, 'response.players.0');
    }

    /**
     * Requests an access token.
     *
     * @param mixed $grant
     * @param mixed[] $options
     * @throws RuntimeException
     */
    public function getAccessToken($grant, array $options = [])
    {
        throw new RuntimeException('Access tokens are not implemented in OpenID.');
    }

    /**
     * Set the Guzzle HTTP client.
     *
     * @param Client $client
     * @return $this
     */
    public function setHttpClient(Client $client)
    {
        $this->client = $client;

        return $this;
    }

    /**
     * Get Guzzle HTTP Client instance.
     *
     * @return Client
     */
    public function getHttpClient()
    {
        if ($this->client !== null) {
            return $this->client;
        }

        return new HttpClient;
    }

    /**
     * Set client ID.
     *
     * @param string $id
     * @return $this
     */
    public function setClientId(string $id)
    {
        return $this->setApiKey($id);
    }

    /**
     * Set API key.
     *
     * @param string $key
     * @return $this
     */
    public function setApiKey(string $key)
    {
        $this->apiKey = $key;

        return $this;
    }

    /**
     * Set client secret.
     *
     * @param string $secret
     * @return void
     */
    public function setClientSecret(string $secret)
    {
        throw new RuntimeException('Unsupported operation.');
    }

    /**
     * Set redirect URI.
     *
     * @param string $redirectUri
     * @return $this
     */
    public function setRedirectUri(string $redirectUri)
    {
        $this->redirectUri = $redirectUri;

        return $this;
    }

    /**
     * Initiation and discovery.
     * https://openid.net/specs/openid-authentication-2_0.html#anchor12
     *
     * @param string $url
     * @return string
     * @throws RuntimeException|GuzzleException
     */
    protected function discover(string $url): string
    {
        $response = $this->client->request('GET', $url);

        $contentType = $response->getHeaderLine('Content-Type');
        $expected    = 'application/xrds+xml;charset=utf-8';
        if (strpos($contentType, $expected) === false) {
            throw new RuntimeException(
                sprintf('Unexpected Content-Type: \'%s\', expected %s', $contentType, $expected)
            );
        }

        $xml = new SimpleXMLElement($response->getBody()->getContents());

        return (string) $xml->XRD->Service->URI;
    }

    /**
     * Assert positive assertion is valid.
     *
     * @return void
     * @throws RuntimeException
     * @throws GuzzleException
     */
    protected function verifyAssertion(): void
    {
        $this->verifyReturnTo();
        $this->verifyNonce();
        $this->verifySignature();
    }

    /**
     * To verify that the "openid.return_to" URL matches the URL that is processing this assertion:
     *   - The URL scheme, authority, and path MUST be the same between the two URLs.
     *   - Any query parameters that are present in the "openid.return_to" URL MUST also be present with the same
     *     values in the URL of the HTTP request the RP received.
     *
     * @return bool
     */
    protected function verifyReturnTo(): bool
    {
        $actual = $this->request->get('openid_return_to');
        $expected = $this->redirectUri;

        if ($actual === $expected) {
            return true;
        }

        throw new RuntimeException(
            vsprintf("openid.return_to value '%s' does not match expected value '%s'", [
                $actual,
                $expected
            ])
        );
    }

    /**
     * Check the nonce is valid.
     *
     * @return bool
     */
    protected function verifyNonce(): bool
    {
        $nonce = new Nonce($this->cache, $this->request->get('openid_response_nonce'));

        return $nonce->isValid();
    }

    /**
     * 11.4.2.  Verifying Directly with the OpenID Provider
     * To have the signature verification performed by the OP, the Relying Party sends a direct request to the OP.
     * To verify the signature, the OP uses a private association that was generated when it issued the positive
     * assertion.
     *
     * @return bool
     * @throws GuzzleException
     */
    protected function verifySignature(): bool
    {
        // If the Claimed Identifier was not previously discovered by the Relying Party (the "openid.identity" in the
        // request was "http://specs.openid.net/auth/2.0/identifier_select" or a different Identifier, or if the OP is
        // sending an unsolicited positive assertion), the Relying Party MUST perform discovery on the Claimed
        // Identifier in the response to make sure that the OP is authorized to make assertions about the Claimed
        // Identifier.
        $claimedIdentifier = $this->request->get('openid_claimed_id');
        $url = $this->discover($claimedIdentifier);

        // Request parameters.
        // Exact copies of all fields from the authentication response, except for "openid.mode".
        $params = $this->request->except('openid_mode');
        $params['openid.mode'] = 'check_authentication';

        $response = $this->client->request('POST', $url, [
            RequestOptions::FORM_PARAMS => $this->replaceUnderscores($params)
        ]);

        $content = $response->getBody()->getContents();
        if (preg_match('/is_valid\s*:\s*true/i', $content)) {
            return true;
        }

        throw new RuntimeException('Invalid signature.');
    }

    /**
     * The returned Claimed ID will contain the user's 64-bit SteamID.
     * The Claimed ID format is: https://steamcommunity.com/openid/id/<steamid>
     *
     * @param string $claimedId
     * @return int|null
     */
    protected function getUserId(string $claimedId): ?int
    {
        $re = '#^https?://steamcommunity.com/openid/id/([0-9]{17,25})#';
        preg_match($re, $claimedId, $matches);

        return is_numeric($matches[1]) ? (int) $matches[1] : null;
    }

    /**
     * PHP converts period (.) to underscore (_) in global variables ($_GET).
     *
     * Support for maintaining periods was only added in Symfony 5.2 which we
     * cannot support yet.
     *
     * @param string[] $params
     * @return string[]
     */
    protected function replaceUnderscores(array $params)
    {
        foreach ($params as $key => $value) {
            $needle = 'openid_';
            if (strpos($key, $needle) === false) {
                continue;
            }

            $params[str_replace($needle, 'openid.', $key)] = $value;
            unset($params[$key]);
        }

        return $params;
    }
}
