<?php declare(strict_types=1);

/**
 * File Twitter.php
 *
 * @copyright  Copyright (c) 2015-2020 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\OAuth\Provider\Twitter;

use Illuminate\Http\RedirectResponse;
use League\OAuth1\Client\Credentials\TokenCredentials;
use SupportPal\OAuth\Exceptions\InvalidArgumentException;
use SupportPal\OAuth\Provider\Provider;
use SupportPal\OAuth\User\OAuth1User;
use SupportPal\OAuth\User\User;

use function array_merge;
use function redirect;
use function str_replace;

/**
 * Class Twitter
 */
class Twitter extends Provider
{
    /**
     * Session key for temporary credentials.
     */
    const TEMPORARY_CREDENTIALS = 'oauth.temp';

    /**
     * Redirect to the OAuth provider.
     *
     * @param string $action
     * @param string[] $options
     * @return RedirectResponse
     */
    public function redirect(string $action, array $options = [])
    {
        // Store the name of the action we want to call after the OAuth flow.
        $this->storeAction($action);

        // Retrieve temporary credentials
        $temporaryCredentials = $this->getServer()->getTemporaryCredentials();

        // Store credentials in the session, we'll need them later
        $this->request->session()->put(static::TEMPORARY_CREDENTIALS, $temporaryCredentials);

        // Second part of OAuth 1.0 authentication is to redirect the
        // resource owner to the login screen on the server.
        $url = $this->getServer()->getAuthorisationUrl([$temporaryCredentials]);

        return redirect()->to($url);
    }

    /**
     * Get the User instance for the authenticated user.
     *
     * @return User
     */
    public function user()
    {
        if (! $this->hasNecessaryVerifier()) {
            throw new InvalidArgumentException('Invalid request. Missing OAuth verifier.');
        }

        $user = $this->getServer()->getUserDetails($token = $this->getToken(), true);

        return (new OAuth1User)
            ->setRaw(array_merge($user->extra, $user->urls))
            ->setToken($token->getIdentifier())
            ->setTokenSecret($token->getSecret())
            ->map([
                'id'              => $user->uid,
                'nickname'        => $user->nickname,
                'name'            => $user->name,
                'email'           => $user->email,
                'avatar'          => $user->imageUrl,
                'avatar_original' => str_replace('_normal', '', $user->imageUrl),
            ]);
    }

    /**
     * League OAuth instance.
     *
     * @return Server
     */
    public function getServer(): Server
    {
        return $this->server;
    }

    /**
     * Determine if the request has the necessary OAuth verifier.
     *
     * @return bool
     */
    protected function hasNecessaryVerifier()
    {
        return $this->request->has('oauth_token') && $this->request->has('oauth_verifier');
    }

    /**
     * Get the token credentials for the request.
     *
     * @return TokenCredentials
     */
    protected function getToken()
    {
        $temp = $this->request->session()->get(static::TEMPORARY_CREDENTIALS);

        return $this->getServer()->getTokenCredentials(
            $temp,
            $this->request->get('oauth_token'),
            $this->request->get('oauth_verifier')
        );
    }
}
