<?php declare(strict_types=1);

namespace SupportPal\OAuth\Provider\Whmcs;

use League\OAuth2\Client\Provider\AbstractProvider;
use League\OAuth2\Client\Provider\Exception\IdentityProviderException;
use League\OAuth2\Client\Token\AccessToken;
use Psr\Http\Message\ResponseInterface;
use SupportPal\OAuth\Provider\Server as Contract;

use function http_build_query;
use function json_decode;

class Server extends AbstractProvider implements Contract
{
    protected ?string $url;

    /**
     * Get authorization url to begin OAuth flow
     *
     * @return string
     */
    public function getBaseAuthorizationUrl()
    {
        return $this->getOpenIdConfig()['authorization_endpoint'];
    }

    /**
     * Get access token url to retrieve token
     *
     * @param mixed[] $params
     * @return string
     */
    public function getBaseAccessTokenUrl(array $params)
    {
        return $this->getOpenIdConfig()['token_endpoint'];
    }

    /**
     * Get default scopes
     *
     * @return array
     */
    protected function getDefaultScopes()
    {
        return ['openid', 'email', 'profile'];
    }

    /**
     * Check a provider response for errors.
     *
     * @param ResponseInterface $response
     * @param mixed[] $data
     * @return void
     * @throws IdentityProviderException
     */
    protected function checkResponse(ResponseInterface $response, $data)
    {
        if (isset($data['error'])) {
            throw new IdentityProviderException(
                $data['error_description'] ?? $response->getReasonPhrase(),
                $response->getStatusCode(),
                $response
            );
        }
    }

    /**
     * Generate a user object from a successful user details request.
     *
     * @param mixed[] $response
     * @param AccessToken $token
     * @return ResourceOwner
     */
    protected function createResourceOwner(array $response, AccessToken $token)
    {
        return new ResourceOwner($response);
    }

    /**
     * Get provider url to fetch user details
     *
     * @param  AccessToken $token
     * @return string
     */
    public function getResourceOwnerDetailsUrl(AccessToken $token)
    {
        return $this->getOpenIdConfig()['userinfo_endpoint']
            . '?' . http_build_query(['access_token' => $token->getToken()]);
    }

    /**
     * Returns the string that should be used to separate scopes when building
     * the URL for requesting an access token.
     *
     * @return string Scope separator, defaults to ','
     */
    protected function getScopeSeparator()
    {
        return ' ';
    }

    /**
     * @return mixed[]
     */
    protected function getOpenIdConfig(): array
    {
        static $data = null;

        if ($data === null) {
            $configUrl = $this->url . '/oauth/openid-configuration.php';

            $response = $this->getHttpClient()->get($configUrl);

            $data = json_decode((string) $response->getBody(), true);
        }

        return $data;
    }

    /**
     * Set client ID.
     *
     * @param string $id
     * @return $this
     */
    public function setClientId(string $id): self
    {
        $this->clientId = $id;

        return $this;
    }

    /**
     * Set client secret.
     *
     * @param string $secret
     * @return $this
     */
    public function setClientSecret(string $secret): self
    {
        $this->clientSecret = $secret;

        return $this;
    }

    /**
     * Set redirect URI.
     *
     * @param string $redirectUri
     * @return $this
     */
    public function setRedirectUri(string $redirectUri): self
    {
        $this->redirectUri = $redirectUri;

        return $this;
    }

    /**
     * Builds the authorization URL.
     *
     * @param string[] $options
     * @return string
     */
    public function getAuthorisationUrl(array $options = []): string
    {
        return $this->getAuthorizationUrl($options);
    }

    /**
     * Set URL.
     *
     * @param string $url
     * @return $this
     */
    public function setUrl(string $url): self
    {
        $this->url = $url;

        return $this;
    }
}
