<?php declare(strict_types=1);

/**
 * File Config.php
 *
 * @copyright  Copyright (c) 2015-2020 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\Purifier\Config;

use HTMLPurifier_Config;
use HTMLPurifier_Exception;
use Illuminate\Config\Repository;

use function array_key_exists;
use function is_dir;
use function mkdir;

/**
 * Class Config
 */
class Config
{
    /**
     * Laravel config repository.
     *
     * @var Repository
     */
    protected $config;

    /**
     * Config constructor.
     *
     * @param Repository $config
     * @throws HTMLPurifier_Exception
     */
    public function __construct(Repository $config)
    {
        if (! $config->has('purifier')) {
            throw new HTMLPurifier_Exception('Configuration parameters not loaded!');
        }

        $this->config = $config;

        // Create cache directory if it doesn't exist.
        $this->initCache($config->get('purifier.cachePath'));
    }

    /**
     * Make HTMLPurifier Config instance.
     *
     * @param array<string>|null $config
     * @return HTMLPurifier_Config
     */
    public function make(?array $config = null)
    {
        $config = $this->merge($config);

        // Remove HTML5 specific rules.
        foreach (Html5Config::RULES as $rule) {
            if (! array_key_exists($rule, $config)) {
                continue;
            }

            unset($config[$rule]);
        }

        return HTMLPurifier_Config::create($config);
    }

    /**
     * Merge the given config with the default.
     *
     * @param array<string>|null $config
     * @return array
     */
    protected function merge(?array $config = null)
    {
        if ($config === null) {
            $config = [];
        }

        return $config + $this->default();
    }

    /**
     * Default purifier config.
     *
     * @return array
     */
    protected function default()
    {
        return [
            'Core.Encoding'        => $this->config->get('purifier.encoding'),
            'Cache.SerializerPath' => $this->config->get('purifier.cachePath'),
        ] + $this->config->get('purifier.settings.default');
    }

    /**
     * Create cache directory if it doesn't exist.
     *
     * @param  string $path
     * @return bool
     */
    protected function initCache($path)
    {
        if (! is_dir($path)) {
            return mkdir($path, 0755, true);
        }

        return true;
    }
}
