<?php declare(strict_types=1);

/**
 * File Purifier.php
 *
 * @copyright  Copyright (c) 2015-2020 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\Purifier;

use HTMLPurifier;
use HTMLPurifier_Config;
use SupportPal\Purifier\Config\Config;
use SupportPal\Purifier\Twig\Replacer;

use function array_map;
use function is_array;

/**
 * Class Purifier
 */
class Purifier
{
    /** @var Config */
    protected $config;

    /** @var HTMLPurifier */
    protected $purifier;

    /**
     * Constructor
     *
     * @param Config $config
     */
    public function __construct(Config $config)
    {
        $this->config = $config;
        $this->purifier = new HTMLPurifier($this->config);
    }

    /**
     * Remove malicious HTML.
     *
     * @param array<string>|string $dirty
     * @param array<mixed> $config
     * @return mixed
     */
    public function clean($dirty, ?array $config = null)
    {
        if (is_array($dirty)) {
            return array_map(function ($item) use ($config) {
                return $this->clean($item, $config);
            }, $dirty);
        }

        return $this->getInstance()->purify($dirty, $this->config($config));
    }

    /**
     * Run HTMLPurifier over a twig template.
     *
     * @param  string  $dirty
     * @param  array<string>   $config
     * @return string
     */
    public function cleanTwig($dirty, ?array $config = null)
    {
        $replacer = new Replacer($dirty);

        return $replacer->run(function ($string) use ($config) {
            return $this->clean($string, $config);
        });
    }

    /**
     * Get HTMLPurifier instance.
     *
     * @return HTMLPurifier
     */
    public function getInstance()
    {
        return $this->purifier;
    }

    /**
     * If you pass a custom configuration to the HTMLPurifier purify function
     * it replaces the preconfigured config instance rather than merges, see
     * comment:
     *  > // make the config merge in, instead of replace
     *
     * If a custom configuration is provided, this ensures that it's merged in
     * with the defaults.
     *
     * @param array<mixed>|null $config
     * @return HTMLPurifier_Config
     */
    public function config(?array $config = null): HTMLPurifier_Config
    {
        return $this->config->make($config);
    }
}
