<?php declare(strict_types=1);

/**
 * File Replacer.php
 *
 * @copyright  Copyright (c) 2015-2020 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\Purifier\Twig;

use Illuminate\Support\Str;

use function call_user_func;
use function preg_replace_callback;
use function str_replace;

/**
 * Class Replacer
 */
class Replacer
{
    /**
     * Regex to find twig code.
     *
     * @var string[]
     */
    protected $patterns = [
        "/(?<=src=[\"'])(data:[^\\/\\s]*\\/[^;\\s]*;base64,\\{\\{(?!%)\\s*(?:'[^']+'|[^}\"]+)\\s*(?<!%)\\}\\})(?=[\"'])/",
        '/(\{\{(?!%)\s*(?:\'[^\']+\'|[^}]+)\s*(?<!%)\}\})/',
        '/(\{\%(?!{)[\s\S]+?(?<!})\%\})/',
    ];

    /**
     * The HTML content.
     *
     * @var string
     */
    protected $string;

    /**
     * List of replacements.
     *
     * @var array
     */
    protected $replacements = [];

    /**
     * Replacer constructor.
     *
     * @param string $string
     */
    public function __construct(string $string = '')
    {
        $this->string = $string;
    }

    /**
     * Remove all twig code before running the string through the callback, and then add it all
     * back afterwards.
     */
    public function run(callable $callback): string
    {
        $string = call_user_func($callback, $this->replaceTwig());

        return $this->replaceRefs($string);
    }

    /**
     * Replace twig code with temporary placeholders.
     */
    protected function replaceTwig(): string
    {
        return preg_replace_callback(
            $this->patterns,
            function ($matches) use (&$originals) {
                $uniq_id = Str::random(64);
                $this->replacements[$uniq_id] = $matches[0];

                return $uniq_id;
            },
            $this->string
        );
    }

    /**
     * Replace the temporary placeholders with the original twig code.
     */
    protected function replaceRefs(string $string): string
    {
        // Add the merge fields back.
        foreach ($this->replacements as $placeholder => $original) {
            $string = str_replace($placeholder, $original, $string);
        }

        return $string;
    }
}
