<?php declare(strict_types=1);

use Soundasleep\Html2Text;
use Soundasleep\Html2TextException;

if (! function_exists('mb_ucfirst')) {
    /**
     * Multi-byte version of first letter to upper-case
     *
     * @param  string $str
     * @return  string
     */
    function mb_ucfirst($str)
    {
        $fc = mb_strtoupper(mb_substr($str, 0, 1), 'UTF-8');

        return $fc . mb_substr($str, 1);
    }
}

if (! function_exists('mb_lcfirst')) {
    /**
     * Multi-byte version of first letter to lower-case
     *
     * @param  string $str
     * @return  string
     */
    function mb_lcfirst($str)
    {
        $fc = mb_strtolower(mb_substr($str, 0, 1), 'UTF-8');

        return $fc . mb_substr($str, 1);
    }
}

if (! function_exists('html2text')) {
    /**
     * Convert a HTML string to text.
     *
     * @param  string $html
     * @param  mixed[] $options
     * @return string
     */
    function html2text($html, array $options = [])
    {
        try {
            // We need to ensure that $html is UTF-8, otherwise we start to get mangled character encoding
            // from the output. For example, if I send $html in to the function UTF-8 encoding and run loadHTML()
            // the function will automatically convert it to ISO-8859-1
            // See: http://stackoverflow.com/questions/8218230/php-domdocument-loadhtml-not-encoding-utf-8-correctly
            $html = mb_convert_encoding($html, 'HTML-ENTITIES', 'UTF-8');

            // In case the string is empty, return as it is.
            if (empty($html)) {
                return $html;
            }

            // Disable DOMDocument warnings bubbling up to errors.
            libxml_use_internal_errors(true);

            // Run the HTML 2 text conversion & fix any encoding problems.
            return Html2Text::convert($html, $options);
        } catch (Html2TextException $e) {
            // Fall back to suboptimal approach...
            return strip_tags($html);
        } finally {
            // Clear the errors, ready for the next run of the function
            libxml_clear_errors();
        }
    }
}

if (! function_exists('is_image')) {
    /**
     * Check whether a given URL points to an image file.
     *
     * @param  string $url
     * @return bool
     * @deprecated 2.1.0 Was used in logo validation but removed as fopen is not always allowed
     */
    function is_image($url)
    {
        $params = ['http' => ['method' => 'HEAD']];
        $ctx = stream_context_create($params);
        $fp = @fopen($url, 'rb', false, $ctx);
        if (! $fp) {
            return false;  // Problem with url
        }

        $meta = stream_get_meta_data($fp);
        if (! isset($meta['wrapper_data'])) {
            fclose($fp);

            return false;  // Problem reading data from url
        }

        $wrapper_data = $meta['wrapper_data'];
        if (is_array($wrapper_data)) {
            foreach (array_keys($wrapper_data) as $hh) {
                // strlen("Content-Type: image") == 19
                if (substr($wrapper_data[$hh], 0, 19) === 'Content-Type: image') {
                    fclose($fp);

                    return true;
                }
            }
        }

        fclose($fp);

        return false;
    }
}

if (! function_exists('isValidTimeStamp')) {
    /**
     * Determine whether a string is a valid timestamp.
     *
     * @param string $timestamp
     * @return bool
     */
    function isValidTimeStamp(string $timestamp)
    {
        return ((string) (int) $timestamp === $timestamp)
            && ($timestamp <= PHP_INT_MAX)
            && ($timestamp >= ~PHP_INT_MAX);
    }
}

if (! function_exists('hours')) {
    /**
     * A list of hours, padded with a zero if less than 10
     *
     * @return array<string>
     */
    function hours()
    {
        $hours = [];
        for ($i = 0; $i < 24; $i++) {
            // Add each hour to array
            $hours[$i] = str_pad((string) $i, 2, '0', STR_PAD_LEFT);
        }

        return $hours;
    }
}

if (! function_exists('randomLetter')) {
    /**
     * Function to generate a single random letter
     *
     * @return string Random letter from uppercase alphabet
     */
    function randomLetter()
    {
        return strtoupper(chr(mt_rand(97, 122)));
    }
}

if (! function_exists('randomNumber')) {
    /**
     * Function to generate a single random digit
     *
     * @return int A random digit (0 - 9);
     */
    function randomNumber()
    {
        return rand(0, 9);
    }
}

if (! function_exists('getSlug')) {
    /**
     * Function to get the slug of a string.
     *
     * @param string $string
     * @return mixed|string
     */
    function getSlug($string)
    {
        $string = strip_tags($string);
        // Preserve escaped octets.
        $string = (string) preg_replace('|%([a-fA-F0-9][a-fA-F0-9])|', '---$1---', $string);
        // Remove percent signs that are not part of an octet.
        $string = str_replace('%', '', $string);
        // Restore octets.
        $string = (string) preg_replace('|---([a-fA-F0-9][a-fA-F0-9])---|', '%$1', $string);

        if (seems_utf8($string)) {
            if (function_exists('mb_strtolower')) {
                $string = mb_strtolower($string, 'UTF-8');
            }

            $string = utf8_uri_encode($string);
        }

        $string = mb_strtolower($string, 'UTF-8');
        $string = (string) preg_replace('/&.+?;/', '', $string); // kill entities
        $string = str_replace('.', '-', $string);
        $string = (string) preg_replace('/[^%a-z0-9 _-]/', '', $string);
        $string = (string) preg_replace('/\s+/', '-', $string);
        $string = (string) preg_replace('|-+|', '-', $string);
        $string = trim($string, '-');

        return $string;
    }
}

if (! function_exists('seems_utf8')) {
    /**
     * Checks if a string looks like it's UTF-8 formatted
     *
     * @param string $str
     * @return bool
     */
    function seems_utf8($str)
    {
        $length = strlen($str);
        for ($i = 0; $i < $length; $i++) {
            $c = ord($str[$i]);
            if ($c < 0x80) {
                $n = 0;  # 0bbbbbbb
            } elseif (($c & 0xE0) === 0xC0) {
                $n = 1; # 110bbbbb
            } elseif (($c & 0xF0) === 0xE0) {
                $n = 2; # 1110bbbb
            } elseif (($c & 0xF8) === 0xF0) {
                $n = 3; # 11110bbb
            } elseif (($c & 0xFC) === 0xF8) {
                $n = 4; # 111110bb
            } elseif (($c & 0xFE) === 0xFC) {
                $n = 5; # 1111110b
            } else {
                return false; # Does not match any model
            }

            for ($j = 0; $j < $n; $j++) { # n bytes matching 10bbbbbb follow ?
                if ((++$i === $length) || ((ord($str[$i]) & 0xC0) !== 0x80)) {
                    return false;
                }
            }
        }

        return true;
    }
}

if (! function_exists('utf8_uri_encode')) {
    /**
     * Encodes a UTF-8 string to URI format
     *
     * @param string $utf8_string
     * @param int $length
     * @return string
     */
    function utf8_uri_encode($utf8_string, $length = 0)
    {
        $unicode = '';
        $values = [];
        $num_octets = 1;
        $unicode_length = 0;

        $string_length = strlen($utf8_string);
        for ($i = 0; $i < $string_length; $i++) {
            $value = ord($utf8_string[$i]);

            if ($value < 128) {
                if ($length && ($unicode_length >= $length)) {
                    break;
                }

                $unicode .= chr($value);
                $unicode_length++;
            } else {
                if (count($values) === 0) {
                    $num_octets = $value < 224 ? 2 : 3;
                }

                $values[] = $value;

                if ($length && ($unicode_length + ($num_octets * 3)) > $length) {
                    break;
                }

                if (count($values) === $num_octets) {
                    if ($num_octets === 3) {
                        $unicode .= '%' . dechex($values[0]) . '%' . dechex($values[1]) . '%' . dechex($values[2]);
                        $unicode_length += 9;
                    } else {
                        $unicode .= '%' . dechex($values[0]) . '%' . dechex($values[1]);
                        $unicode_length += 6;
                    }

                    $values = [];
                    $num_octets = 1;
                }
            }
        }

        return $unicode;
    }
}

if (! function_exists('isRegexSafe')) {
    /**
     * Whether or not a regular expression is safe to use
     *
     * @param  string $regex Regular expression
     * @return boolean       TRUE on success, FALSE on error
     */
    function isRegexSafe($regex)
    {
        // null character allows a premature regex end and "/../e" injection
        if (! is_string($regex) || strpos($regex, chr(0)) !== false || ! trim($regex)) {
            return false;
        }

        $backtrack_limit = ini_set('pcre.backtrack_limit', '200');
        $recursion_limit = ini_set('pcre.recursion_limit', '20');

        $valid = @preg_match($regex, '') !== false;

        if (is_string($backtrack_limit)) {
            ini_set('pcre.backtrack_limit', $backtrack_limit);
        }

        if (is_string($recursion_limit)) {
            ini_set('pcre.recursion_limit', $recursion_limit);
        }

        return $valid;
    }
}

if (! function_exists('escape_like')) {
    /**
     * Escape the % character in SQL LIKE statements.
     */
    function escape_like(string $string): string
    {
        $search = ['%', '_'];
        $replace = ['\%', '\_'];

        return str_replace($search, $replace, $string);
    }
}

if (! function_exists('str_lreplace')) {
    /**
     * Replace the last occurrence of a string in a string
     *
     * @param  string $search  String to search for
     * @param  string $replace What to replace the string with
     * @param  string $subject The subject to search
     * @return string
     */
    function str_lreplace($search, $replace, $subject)
    {
        $pos = strrpos($subject, $search);

        if ($pos !== false) {
            $subject = substr_replace($subject, $replace, $pos, strlen($search));
        }

        return $subject;
    }
}

if (! function_exists('addOrdinalNumberSuffix')) {
    /**
     * Add ordinal number suffix to a numeric value
     *
     * @param int $num
     * @return string
     */
    function addOrdinalNumberSuffix($num)
    {
        if (! in_array($num % 100, [11, 12, 13])) {
            switch ($num % 10) {
                // Handle 1st, 2nd, 3rd
                case 1:
                    return $num . 'st';

                case 2:
                    return $num . 'nd';

                case 3:
                    return $num . 'rd';
            }
        }

        return $num . 'th';
    }
}

if (! function_exists('str_contains_letters')) {
    /**
     * Check whether a string contains one or more alpha characters.
     * Used by password strength validation.
     *
     * @param  string $value
     * @return bool
     */
    function str_contains_letters($value)
    {
        return preg_match('/\pL/', $value) === 1;
    }
}

if (! function_exists('str_contains_digits')) {
    /**
     * Check whether a string contains one or more numbers.
     * Used by password strength validation.
     *
     * @param  string $value
     * @return bool
     */
    function str_contains_digits($value)
    {
        return preg_match('/\pN/', $value) === 1;
    }
}

if (! function_exists('str_contains_case')) {
    /**
     * Check whether a string contains both lowercase and uppercase characters.
     * Used by password strength validation.
     *
     * @param  string $value
     * @return bool
     */
    function str_contains_case($value)
    {
        return preg_match('/(\p{Ll}+.*\p{Lu})|(\p{Lu}+.*\p{Ll})/u', $value) === 1;
    }
}

if (! function_exists('str_contains_symbols')) {
    /**
     * Check whether a string contains one or more symbols.
     * Used by password strength validation.
     *
     * @param  string $value
     * @return bool
     */
    function str_contains_symbols($value)
    {
        return preg_match('/[!@#$%^&*?()\-_=+{};:,<.>]/', $value) === 1;
    }
}

if (! function_exists('is_gt')) {
    /**
     * Check whether x is greater than y.
     *
     * @param  int $x
     * @param  int $y
     * @return bool
     */
    function is_gt($x, $y)
    {
        return $x > $y;
    }
}

if (! function_exists('is_lt')) {
    /**
     * Check whether x is less than y.
     *
     * @param  int $x
     * @param  int $y
     * @return bool
     */
    function is_lt($x, $y)
    {
        return $x < $y;
    }
}

if (! function_exists('array_unique_ci')) {
    /**
     * Case-insensitive array_unique implementation.
     *
     * @param string[] $array
     * @return array<string>
     */
    function array_unique_ci(array $array)
    {
        return array_intersect_key(
            $array,
            array_unique(array_map('strtolower', $array))
        );
    }
}

if (! function_exists('strposa')) {
    /**
     * Check whether a haystack contains at least one needle.
     *
     * @param  string $haystack
     * @param  string[] $needles
     * @param  int    $offset
     * @return int|false
     */
    function strposa($haystack, $needles = [], $offset = 0)
    {
        $chr = [];

        foreach ($needles as $needle) {
            $res = strpos($haystack, $needle, $offset);

            if ($res === false) {
                continue;
            }

            $chr[$needle] = $res;
        }

        return empty($chr) ? false : min($chr);
    }
}

if (! function_exists('of_type')) {
    /**
     * Check whether an object matches one of the provided classes.
     *
     * @param  object $object        The tested object
     * @param  string $instances,... List of classes
     * @return bool
     */
    function of_type($object, $instances)
    {
        $num_args = func_num_args();
        $arg_list = func_get_args();
        for ($i = 1; $i < $num_args; $i++) {
            if ($object instanceof $arg_list[$i]) {
                return true;
            }
        }

        return false;
    }
}

if (! function_exists('dom_document')) {
    /**
     * Load a DomDocument instance.
     *
     * @param  string $html
     * @return DOMDocument
     */
    function dom_document($html)
    {
        $dom = new DOMDocument;

        // Hide errors
        libxml_use_internal_errors(true);

        // We need to ensure that $html is UTF-8, otherwise we start to get mangled character encoding
        // from the output. For example, if I send $html in to the function UTF-8 encoding and run loadHTML()
        // the function will automatically convert it to ISO-8859-1
        // See: http://stackoverflow.com/questions/8218230/php-domdocument-loadhtml-not-encoding-utf-8-correctly
        @$dom->loadHTML(mb_convert_encoding($html, 'HTML-ENTITIES', 'UTF-8'), LIBXML_PARSEHUGE);
        $dom->preserveWhiteSpace = false;
        $dom->formatOutput = true;

        return $dom;
    }
}

if (! function_exists('in_array_ci')) {
    /**
     * Checks if a value exists in an array with a case insensitive search.
     *
     * @param  string $needle
     * @param  string[] $haystack
     * @param  bool  $strict [optional]
     * @return bool true if needle is found in the array, false otherwise.
     */
    function in_array_ci($needle, $haystack, $strict = false)
    {
        return in_array(strtolower($needle), array_map('strtolower', $haystack), $strict);
    }
}

if (! function_exists('array_search_ci')) {
    /**
     * Searches the array for a given value and returns the corresponding key if successful.
     *
     * @param  string $needle
     * @param  string[] $haystack
     * @param  bool  $strict [optional]
     * @return mixed|false true if needle is found in the array, false otherwise.
     */
    function array_search_ci($needle, $haystack, $strict = false)
    {
        return array_search(strtolower($needle), array_map('strtolower', $haystack), $strict);
    }
}

if (! function_exists('format_bytes')) {
    /**
     * Convert bytes to human readable format.
     *
     * @param  int $size
     * @param  int $precision
     * @return string
     */
    function format_bytes($size, $precision = 2)
    {
        $base = log($size, 1024);
        $suffixes = ['', 'kB', 'MB', 'GB', 'TB'];

        return round(pow(1024, $base - floor($base)), $precision) . ' ' . $suffixes[floor($base)];
    }
}

if (! function_exists('to_bytes')) {
    /**
     * Converts a human readable file size value to a number of bytes that it
     * represents. Supports the following modifiers: K, M, G and T.
     * Invalid input is returned unchanged.
     *
     * Example:
     * <code>
     * $config->human2byte(10);          // 10
     * $config->human2byte('10b');       // 10
     * $config->human2byte('10k');       // 10240
     * $config->human2byte('10K');       // 10240
     * $config->human2byte('10kb');      // 10240
     * $config->human2byte('10Kb');      // 10240
     * // and even
     * $config->human2byte('   10 KB '); // 10240
     * </code>
     */
    function to_bytes(string $value): ?string
    {
        return preg_replace_callback('/^\s*(\d+)\s*(?:([kmgt]?)b?)?\s*$/i', function ($m) {
            $m1 = (int) $m[1];
            switch (strtolower($m[2])) {
                case 't':
                    $m1 *= 1024;
                // no break
                case 'g':
                    $m1 *= 1024;
                // no break
                case 'm':
                    $m1 *= 1024;
                // no break
                case 'k':
                    $m1 *= 1024;
            }

            return (string) $m1;
        }, $value);
    }
}

if (! function_exists('decode_json_array')) {
    /**
     * Decode a JSON encoded array.
     *
     * @param  string|null $value
     * @return array<mixed>
     */
    function decode_json_array(?string $value): array
    {
        // Convert json encoded string to array
        if ($value !== null && is_array($json = json_decode($value, true))) {
            return $json;
        }

        return [];
    }
}

if (! function_exists('convert_ip_to_in_addr')) {
    /**
     * Convert IP address to packed in_addr representation.
     *
     * @param string $address
     * @return string|null
     */
    function convert_ip_to_in_addr($address)
    {
        if (filter_var($address, FILTER_VALIDATE_IP) && ($ip = @inet_pton($address)) !== false) {
            return $ip;
        }

        return null;
    }
}

if (! function_exists('http_parse_headers')) {
    /**
     * Parses HTTP headers into an associative array.
     *
     * Replicates http_parse_headers from pecl_http, code inspired by https://github.com/rmccue/Requests
     *
     * @param  string $raw_headers
     * @return array<string>|array<string[]>
     * @see https://secure.php.net/manual/pl/function.http-parse-headers.php
     */
    function http_parse_headers($raw_headers)
    {
        $return = [];

        // Pretend CRLF = LF for compatibility (RFC 2616, section 19.3)
        $headers = str_replace("\r\n", "\n", $raw_headers);

        // Unfold headers (replace [CRLF] 1*( SP | HT ) with SP) as per RFC 2616 (section 2.2)
        $headers = (string) preg_replace('/\n[ \t]/', ' ', $headers);

        foreach (explode("\n", $headers) as $header) {
            [$key, $value] = array_pad(explode(':', $header, 2), 2, null);
            if (empty($key) || $value === null) {
                continue;
            }

            // Remove trailing space.
            $value = trim($value);

            // Duplicate headers should be an array.
            $return[$key] = isset($return[$key]) ? array_merge((array) $return[$key], [$value]) : $value;
        }

        return $return;
    }
}

if (! function_exists('guess_mimetype')) {
    /**
     * Guess the mime type of a string.
     *
     * @param  string $string
     * @param  string $default
     * @return string
     */
    function guess_mimetype($string, $default = 'text/plain')
    {
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mime_type = $finfo->buffer($string);

        return $mime_type !== false ? $mime_type : $default;
    }
}
