<?php declare(strict_types=1);

namespace SupportPal\Support\Archive;

use Psr\Log\LoggerInterface;
use SupportPal\Support\Exception\TarProcessException;
use SupportPal\Support\Files\ExecutableFinder;
use SupportPal\Support\Process\ProcessFactory;
use SupportPal\Support\Process\ProcessHandler;
use SupportPal\Support\Repository\SettingsRepository;
use SupportPal\Support\Utils\Storage;
use Symfony\Component\Process\Process;

use function array_map;
use function array_merge;
use function array_splice;
use function basename;
use function dirname;
use function realpath;
use function sprintf;
use function strpos;

class TarArchiveManager implements ArchiveManager
{
    private ExecutableFinder $executableFinder;
    private ProcessFactory $processFactory;

    private ?string $tarPath = null;

    private Storage $storage;

    private ProcessHandler $processHandler;

    private ?LoggerInterface $logger = null;

    private SettingsRepository $settingsRepository;

    public function __construct(
        ProcessFactory $processFactory,
        ExecutableFinder $executableFinder,
        Storage $storage,
        ProcessHandler $processHandler,
        SettingsRepository $settingsRepository
    ) {
        $this->processFactory = $processFactory;
        $this->executableFinder = $executableFinder;
        $this->storage = $storage;
        $this->processHandler = $processHandler;
        $this->settingsRepository = $settingsRepository;
    }

    public function isGnuTar(): bool
    {
        $process = $this->handleProcess([$this->getTarBinaryPath(), '--help']);

        return strpos($process->getOutput(), '--force-local') !== false;
    }

    /**
     * @throws TarProcessException
     */
    public function extract(string $source, string $target, bool $isCompressed): void
    {
        /** @var string $sourceAbsolutePath */
        $sourceAbsolutePath = realpath($source);
        $command = [
            $this->getTarBinaryPath(),
            $isCompressed ? '-xzf' : '-xf',
            $sourceAbsolutePath,
            '-C',
            $this->storage->normalize($target)
        ];

        if ($this->isGnuTar()) {
            array_splice($command, 1, 0, '--force-local');
        }

        $this->handleProcess($command, $sourceAbsolutePath);
    }

    /**
     * @param string[] $files
     */
    public function archive(string $target, array $files, bool $removeFiles = true): void
    {
        $dir = dirname($target);

        $command = [$this->getTarBinaryPath(), '-cf', $target, '-C', $this->storage->normalize($dir)];
        if ($this->isGnuTar()) {
            array_splice($command, 1, 0, '--force-local');
        }

        $command = array_merge($command, $this->getFilesBasename($files));
        $this->handleProcess($command, $target);

        // --remove-files is a GNU Tar extension and is unavailable in BSD Tar.
        if (! $removeFiles) {
            return;
        }

        array_map('unlink', $files);
    }

    public function getTarPath(): string
    {
        if ($this->tarPath === null) {
            return $this
                ->setTarPath($this->settingsRepository->getTarPath() ?? '')
                ->getTarPath();
        }

        return $this->tarPath;
    }

    public function setTarPath(string $path): self
    {
        $this->tarPath = $path;

        return $this;
    }

    public function supports(string $mimeType): bool
    {
        return $mimeType === 'application/gzip' || $mimeType === 'application/x-tar';
    }

    /**
     * @param string[] $command
     */
    private function handleProcess(array $command, ?string $filePath = null): Process
    {
        $process = $this->processFactory->create($command);
        if ($this->logger !== null) {
            $this->processHandler->setLogger($this->logger);
        }

        $process = $this->processHandler->handle($process, $this->storage->basePath());
        if ($process->getExitCode() !== 0) {
            throw new TarProcessException(
                $filePath,
                sprintf('Command %s, Failed with Output: %s.', $process->getCommandLine(), $process->getErrorOutput())
            );
        }

        return $process;
    }

    /**
     * @param string[] $files
     * @return string[]
     */
    private function getFilesBasename(array $files): array
    {
        return array_map(function (string $filename) {
            return basename($filename);
        }, $files);
    }

    private function getTarBinaryPath(): string
    {
        if (! empty($tar = $this->getTarPath())) {
            return $tar;
        }

        return $this->executableFinder->find('tar');
    }
}
