<?php declare(strict_types=1);

namespace SupportPal\Support\Command;

use SupportPal\Support\Files\Filesystem;
use SupportPal\Support\Process\Process;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

use function base_path;
use function is_string;
use function trans;

abstract class Command extends \Illuminate\Console\Command
{
    /**
     * Create a new console command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();

        $this->getDefinition()->addOptions([
            new InputOption('locale', null, InputOption::VALUE_OPTIONAL, 'Locale to show the output in.'),
            new InputOption('suppress-risk', null, InputOption::VALUE_NONE, 'Suppress file permission notices.'),
        ]);
    }

    /**
     * Execute the console command.
     *
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->setLocale()->warnBadRunAsUser();

        /** @var callable $callback */
        $callback = [$this, 'handle'];

        return (int) $this->laravel->call($callback);
    }

    /**
     * Set the locale for the command output.
     *
     * @return $this
     */
    private function setLocale(): self
    {
        $locale = $this->option('locale');
        if (is_string($locale)) {
            $this->laravel->setLocale($locale);
        }

        return $this;
    }

    /**
     * Show a warning if the command is being run by a user that might lead to permission issues.
     *
     * @return self
     */
    private function warnBadRunAsUser(): self
    {
        if ($this->option('suppress-risk') === true) {
            return $this;
        }

        $user = $this->laravel->make(Process::class)->getEffectiveUser();
        $fileOwner = $this->laravel->make(Filesystem::class)->getFileOwnerUsername($path = base_path());

        if (isset($user, $fileOwner) && $user !== $fileOwner) {
            /** @var string $warningMessage */
            $warningMessage = trans('Support::general.possible_permission_issue', ['cli_user' => $user, 'path' => $path, 'path_user' => $fileOwner]);
            $this->warn($warningMessage);
        }

        return $this;
    }
}
