<?php declare(strict_types=1);

namespace SupportPal\Support\Files;

use Illuminate\Support\Str;
use InvalidArgumentException;
use Symfony\Component\Filesystem\Exception\FileNotFoundException;
use Symfony\Component\Filesystem\Exception\IOException;
use Symfony\Component\Filesystem\Filesystem as BaseFilesystem;

use function disk_free_space;
use function disk_total_space;
use function fileowner;
use function function_exists;
use function is_executable;
use function posix_getpwuid;
use function sprintf;
use function sys_get_temp_dir;

class Filesystem
{
    /** @var BaseFilesystem */
    private $filesystem;

    public function __construct(BaseFilesystem $filesystem)
    {
        $this->filesystem = $filesystem;
    }

    /**
     * Lookup which user owns a given path.
     *
     * @param string $path
     * @return string|null
     */
    public function getFileOwnerUsername(string $path): ?string
    {
        if (! function_exists('fileowner') || ! function_exists('posix_getpwuid')) {
            return null;
        }

        $owner = fileowner($path);
        if ($owner === false) {
            return null;
        }

        $pwuid = posix_getpwuid($owner);
        if ($pwuid === false) {
            return (string) $owner;
        }

        return $pwuid['name'] ?? null;
    }

    /**
     * @throws InvalidArgumentException
     */
    public function getFreeSpace(string $path): float
    {
        $space = disk_free_space($path);
        if ($space === false) {
            throw new InvalidArgumentException(sprintf('Unable to determine amount of free space at "%s".', $path));
        }

        return $space;
    }

    /**
     * @throws FileNotFoundException
     * @throws IOException
     */
    public function isReadOnly(string $path): bool
    {
        $target = $path . '.' . Str::random();

        // If we can copy and delete a file, then we don't have a read only file system.
        $this->filesystem->copy($path, $target, true);
        $this->filesystem->remove($target);

        return false;
    }

    /**
     * @throws IOException
     */
    public function makeTempDir(): string
    {
        $path = $this->filesystem->tempnam(sys_get_temp_dir(), 'supportpal-');
        $this->filesystem->remove($path);
        $this->filesystem->mkdir($path);

        return $path;
    }

    /**
     * @throws InvalidArgumentException
     */
    public function getDiskSpace(string $path): float
    {
        $space = disk_total_space($path);
        if ($space === false) {
            throw new InvalidArgumentException(sprintf('Unable to determine amount of space at "%s".', $path));
        }

        return $space;
    }

    public function isExecutable(string $path): bool
    {
        return is_executable($path);
    }

    public function call(): BaseFilesystem
    {
        return $this->filesystem;
    }
}
