<?php declare(strict_types=1);

namespace SupportPal\Support\Files;

use Illuminate\Contracts\Filesystem\Filesystem;
use Illuminate\Filesystem\FilesystemManager;
use Illuminate\Http\Response as HttpResponse;
use League\Flysystem\Adapter\Local as LocalFilesystem;
use League\Flysystem\Filesystem as Flysystem;

abstract class LockFile
{
    private Filesystem $manager;

    protected string $key;

    /**
     * LockFile constructor.
     */
    public function __construct(FilesystemManager $file)
    {
        $this->manager = $file->disk('local');
        $this->key = $this->getKey();
    }

    /**
     * Acquire a lock.
     */
    public function acquire(?string $sessionId = null): bool
    {
        if ($this->isLocked($sessionId)) {
            return false;
        }

        $result = $this->manager->put($this->key, $sessionId ?? '');

        return $result !== false;
    }

    /**
     * Check if the lock file exists.
     */
    public function isLocked(?string $sessionId = null): bool
    {
        if ($this->manager->exists($this->key)) {
            return ! isset($sessionId) || $sessionId !== $this->manager->get($this->key);
        }

        return false;
    }

    /**
     * Return an error response.
     *
     * @return HttpResponse
     */
    abstract public function errorResponse(): HttpResponse;

    /**
     * Release the lock file.
     *
     * @return void
     */
    public function release()
    {
        $this->manager->delete($this->key);
    }

    /**
     * Get the lock file path.
     *
     * @return string
     */
    public function getPath(): string
    {
        /** @var Flysystem $manager */
        $manager = $this->manager;
        /** @var LocalFilesystem $adapter */
        $adapter = $manager->getAdapter();

        return $adapter->applyPathPrefix($this->key);
    }

    abstract protected function getKey(): string;
}
