<?php declare(strict_types=1);

namespace SupportPal\Support\Json;

use Illuminate\Support\Str;
use SplQueue;

use function is_array;
use function is_iterable;
use function json_encode;
use function sprintf;
use function str_replace;

class Json
{
    /** @var mixed */
    private $value;

    /**
     * @param mixed $value
     */
    public function __construct($value)
    {
        $this->value = $value;
    }

    /**
     * @phpstan-param int<1, max> $depth
     * @return array|mixed|string|string[]
     */
    public function encode(int $flags = 0, int $depth = 512)
    {
        // Pre-process and replace javascript expressions with placeholders
        $javascriptExpressions = new SplQueue;
        $valueToEncode = $this->recursiveJsonExprFinder($this->value, $javascriptExpressions);

        // Encoding
        $encodedResult = json_encode($valueToEncode, $flags, $depth);

        // Post-process to revert back any Expr instances.
        return $this->injectJavascriptExpressions($encodedResult, $javascriptExpressions);
    }

    /**
     * @param mixed $value
     * @param SplQueue<mixed> $javascriptExpressions
     * @return array|mixed|string
     */
    private function recursiveJsonExprFinder($value, SplQueue $javascriptExpressions)
    {
        if ($value instanceof Expr) {
            $magicKey = Str::random(64);
            $javascriptExpressions->enqueue(['magicKey' => $magicKey, 'value' => $value]);

            return $magicKey;
        }

        if (is_array($value)) {
            foreach ($value as $k => $v) {
                $value[$k] = $this->recursiveJsonExprFinder($v, $javascriptExpressions);
            }

            return $value;
        }

        if (is_iterable($value)) {
            foreach ($value as $k => $v) {
                $value->$k = $this->recursiveJsonExprFinder($value->$k, $javascriptExpressions);
            }

            return $value;
        }

        return $value;
    }

    /**
     * @param mixed $encodedValue
     * @param SplQueue<mixed> $javascriptExpressions
     * @return array|mixed|string|string[]
     */
    private function injectJavascriptExpressions($encodedValue, SplQueue $javascriptExpressions)
    {
        foreach ($javascriptExpressions as $expression) {
            $encodedValue = str_replace(
                sprintf('"%s"', $expression['magicKey']),
                (string) $expression['value'],
                (string) $encodedValue
            );
        }

        return $encodedValue;
    }
}
