<?php declare(strict_types=1);

namespace SupportPal\Support\Manager;

use GuzzleHttp\Exception\GuzzleException;
use SupportPal\ProxyService\ProxyService;
use SupportPal\Support\Exception\FileNotDownloadedException;
use SupportPal\Support\Exception\FileNotFoundException;
use Symfony\Component\Filesystem\Filesystem;

use function array_replace_recursive;
use function fopen;
use function stream_context_create;

class DownloadManager
{
    private Filesystem $filesystem;

    private ProxyService $proxyService;

    public function __construct(ProxyService $proxyService, Filesystem $filesystem)
    {
        $this->proxyService = $proxyService;
        $this->filesystem = $filesystem;
    }

    /**
     * @throws FileNotFoundException
     */
    public function download(string $url, string $target): string
    {
        try {
            $this->proxyService->createGuzzle()->request('GET', $url, ['sink' => $target]);
        } catch (GuzzleException $exception) {
            // Cleanup file if errored.
            if ($this->filesystem->exists($target)) {
                $this->filesystem->remove($target);
            }

            throw new FileNotDownloadedException($exception->getMessage(), $exception->getCode(), $exception);
        }

        return $target;
    }

    /**
     * @param mixed[] $context
     * @return resource
     */
    public function downloadStream(string $url, array $context = [])
    {
        $context = array_replace_recursive($context, $this->proxyService->streamOpts());
        $resource = fopen($url, 'r', false, stream_context_create($context));
        if ($resource === false) {
            throw new FileNotDownloadedException('Unable to read from ' . $url);
        }

        return $resource;
    }
}
